function [dL,dV,dW] = eigpert(A,L,V,W,dA,p,Err)

% function [dL,dV,dW] = eigpert(A,L,V,W,dA,p,Err)
% 
% This function calculates an estimate of abs(dL) and norm(dV) where
% L+dL are the eigenvalues and V+dV are the eigenvectors of the matrix A+dA
% starting form the approximative eigenvalues L and
% approximative eigenvectors V (with approximative inverse W) of A
%
% p determines the norm
% Err = [Err(1),Err(2)] are the errors in doing multiplications and 
% calculating norms
% No checking of dimensions is done
%
% G. Gripenberg 5.8.1996
% Reference: G. Gripenberg, Computing the joint spectral radius, Linear
%      Algebra Appl.  234 (1996) 43--60.

if nargin < 7
 Err = [];
end

if nargin < 6
  p = 2;
end

n = size(A)*[1;0];

if  size(Err)*[1;1] < 3
  if p==1 | p==inf
    Err = [2^(-50)*n,2^(-49)*n];
  else
    Err = [2^(-50)*n^2,2^(-49)*n^2];
  end
end



dV=0;
dL=zeros(n,1);
if ~(p==1|p==2|p==inf)
  dV=NaN;
  dL=NaN + dL;
  return
end



if size(L)*[1;1]< 2*n
  L=diag(L);
end



VN = norm(V,p)*(1+Err(2));
WN = norm(W,p)*(1+Err(2));

CII = (1-Err(1)*VN*WN-norm(V*W-eye(n),p)*(1+Err(2)));
if CII < eps
 dV=NaN;
 dL=NaN + dL;
 return
end

dW = (Err(1)*VN*WN + norm(W*V-eye(n),p)*(1+Err(2)))*WN/CII;

CP = (dA + norm(A-(V*L)*W,p)*(1+Err(2)) ...
      + norm(V*L,p)*(1+Err(2))*(Err(1)*WN+dW) ...
   + Err(1)*VN*max(abs(diag(L)))*(WN+dW));

diff =inf;

for j = 1:n
   LL = diag(L);
   lambda=LL(j);
   LL(j) = [];
   d = min(abs(lambda-LL));
   PJ = diag((1:n)'==j);
   CA = CP*(norm(PJ*W,p)*(1+Err(2))+ dW)*VN;
   CB = CP*(norm((eye(n)-PJ)*W,p)*(1+Err(2))+ dW)*VN;
   if d <= (sqrt(CA) + sqrt(CB))^2 
     dV=NaN;
     dL(j)=NaN;
   else
     dL(j) = 2*CA*d/(CA-CB+ d + sqrt((CA-CB+ d)^2-4*CA*d));
     diff = min(diff,d-dL(j));
   end
end

CQ = (WN+dW)*VN*CP/diff;

if 1-CQ < eps
 dV=NaN;
 return
end

if dV ~= NaN
  dV = VN*CQ/(1-CQ);
  if p == 2
    dV = sqrt(n)*dV;
  end
end
if nargout > 2
  CII = (1-(Err(1)*VN + dV)*WN-norm(V*W-eye(n),p)*(1+Err(2)));
  dW = ((Err(1)*VN+dV)*WN + norm(W*V-eye(n),p)*(1+Err(2)))*WN/CII;
end


