
/*
 * World.java (part of Propo.java)
 * 
 * Copyright (c) 2002, Matias Dahl 
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

import java.awt.*;
import java.awt.geom.*;
import java.awt.BasicStroke.*;
import java.util.*;

// Application class

//
// important:
//   - always add the special point first to a world.
//
public class World
{
    float F_MIN = (float) java.lang.Float.MIN_VALUE;
    float F_MAX = (float) java.lang.Float.MAX_VALUE;

    // world data
    public PPoint specialPoint;  // special point the user should point at

    java.util.LinkedList worldData;  // circle world


    // -- General routines
    World() 
    {
	worldData = new java.util.LinkedList();

	specialPoint = new PPoint();
    }

    public void add( PObject q )
    {
	worldData.add( q);
    }

    public void add( PPoint q )
    {
	worldData.add((PObject) q);
    }

    // Add a line to the world
    public void add(PLine q)
    {
	worldData.add( (PObject) q );
    }

    // Add circle 
    public void add( PCircle q )
    {
	worldData.add( (PObject) q);
    }

    // Remove all objects from world
    public void clear()
    {
	worldData.clear();
    }

    // return a deep copy of current world. 
    public World Clone()
    {
	World temp = new World();
	Iterator i = worldData.iterator();
	PObject t; 

	while ( i.hasNext() ) 
	    {
		t = (PObject) i.next();
		temp.add( (PObject) t.clone() );
	    }

	temp.setSpecialPoint( (PPoint) this.specialPoint.clone() );

	return ( temp );
    }

    // convert World to user readable string
    public String toString()
    {
	String ans = "\nWorld: \n";
	ListIterator i = worldData.listIterator();
	PObject pObject; 

	ans += "Special point = " + specialPoint.toString() + "\n";

	while ( i.hasNext() ) 
	    {
		pObject = (PObject) i.next();
		ans += "" + i.nextIndex() + " : " + pObject.toString() + "\n";
	    }

	ans += "---- \n";
	
	return ( ans );

    }

    public void print()
    {
	System.out.println( this.toString() ); 
    }


    // ** Routines for special point

    // set special point to q
    public void setSpecialPoint(PPoint q)
    {
	specialPoint = q;
    }


    // ** Distance properties of World

    // return length from a point to the special point
    public float DistanceFromSpecialPoint( PPoint pp )
    {
	return ( pp.distance( specialPoint ) );
    }

    // Measure the shortest distance from pp to any point in world
    // including the special point. 
    public float minDistance( PPoint pp )
    {
	float md, tt;

	md = this.DistanceFromSpecialPoint( pp );
	PObject q;

	ListIterator i = this.worldData.listIterator(); 
	while ( hasNextPoint( i ) ) 
	    {
		q = (PObject) i.next();
		if (q.OType() == Consts.PPOINT)
		    {
			md = Math.min (pp.distance( (PPoint) q), md);
		    }
	    }

	return ( md );
    }

    private PPoint nextPoint( ListIterator i )
    {
	PObject traceObject = null;


	while ( i.hasNext() ) 
	    {
		traceObject = (PObject) i.next();
		if ( traceObject.OType() == Consts.PPOINT) 
		    {
			return ( (PPoint) traceObject );
		    }
	    }

	return ( null );
	
    }

    private boolean hasNextPoint( ListIterator i )
    {	

	PObject traceObject = null;
	int currentIndex = i.nextIndex();

	while ( i.hasNext() ) 
	    {
		traceObject = (PObject) i.next();
		if ( traceObject.OType() == Consts.PPOINT) 
		    {
			while ( i.nextIndex() != currentIndex )
			    {
				i.previous();	

			    }
			return (true);
		    }
	    }

	while ( i.nextIndex() != currentIndex )
	    {
		i.previous();
	    }
	return ( false );
	
    }

    
    // Measure the shortest distance between any two points in the world
    // 
    public float minDistance()
    {
	float ans = (float) F_MAX;
	PObject traceObject;
	ListIterator i = this.worldData.listIterator();

	while (hasNextPoint( i ) )
	    {
		PPoint basePoint = new PPoint( nextPoint( i ), 0f );
		
		ans = Math.min( ans, this.DistanceFromSpecialPoint( basePoint ) );

		int currentIndex = i.nextIndex();

		while ( hasNextPoint( i ) )
		    {
			PPoint runPoint = new PPoint( (PPoint) nextPoint( i ), 0f );			
			ans = Math.min( ans, basePoint.distance( runPoint ) );
		    }

		while ( currentIndex != i.nextIndex() )
		    {
			i.previous();
		    }

	    }
	
	return ( ans );
    }


    // Return coordinates to lower right corner as PPoint
    public PPoint LRCorner()
    {
	PPoint ans = new PPoint( specialPoint, 0f ); 


	PObject tempObject;
     	for (Iterator i = worldData.iterator(); i.hasNext();) 
	    {
		tempObject = (PObject) i.next();
		ans.setX ( Math.max((float) tempObject.getMaxX(), ans.x() ) );
		ans.setY ( Math.max((float) tempObject.getMaxY(), ans.y() ) );
	    }

	return ( ans );
	
    }


    // Return coordinates to upper left corner as PPoint
    public PPoint ULCorner()
    {
	PPoint ans = new PPoint( specialPoint, 0f );


	PObject tempObject;
     	for (Iterator i = worldData.iterator(); i.hasNext();) 
	    {
		tempObject = (PObject) i.next();
		ans.setX ( Math.min((float) tempObject.getMinX(), ans.x() ) );
		ans.setY ( Math.min((float) tempObject.getMinY(), ans.y() ) );
	    }

	return ( ans );
    }


    // -- Operations
    
    // Scale all objects in World by factor
    public void scale (float factor)
    {

	for (Iterator i = worldData.iterator(); i.hasNext();) 
	    {
		PObject q = (PObject) i.next();
		q.scale( factor );
	    }

	specialPoint.scale( factor );
    }


    // Scale all objects in World by factor
    public void rotate (PPoint origin, double angle)
    {
	Iterator i = worldData.iterator(); 

	while (i.hasNext()) 
	    {
		PObject q = (PObject) i.next();
		q.rotate( origin, angle );
	    }

	specialPoint.rotate( origin, angle  );
    }


    // Shift all objects in World by spoint
    public void shift (PPoint spoint)
    {
	for (Iterator i = worldData.iterator(); i.hasNext();) 
	    {
		PObject q = (PObject) i.next();
		q.shift( spoint );
	    }

	specialPoint.shift( spoint );
    }


    // -- Drawing routines

    // Draw whole world
    public void DrawAll (Graphics g)
    {
	this.DrawObjects ( g );
	this.DrawSpecialPoint( g );

    }

    // Draw the special point
    public void DrawSpecialPoint (Graphics g)
    {
	Graphics2D g2 = (Graphics2D) g;
	g2.setRenderingHint( RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON );
	specialPoint.draw ( g2, Color.red );
    }

    // Draw all objects
    public void DrawObjects (Graphics g)
    {
	Graphics2D g2 = (Graphics2D) g;

	g2.setRenderingHint( RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON );

	g2.setStroke( new BasicStroke (2.0f) );
	g2.setColor(Color.black);

	PObject traceObject;

	for (Iterator i = worldData.iterator(); i.hasNext();) 
	    {
		traceObject = (PObject) i.next();
		traceObject.draw( g2, Color.black  );
	    }
    }

}

