
/*
 * PropoGUI.java  (part of Propo.java)
 * 
 * Copyright (c) 2002, Matias Dahl 
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

import javax.swing.*;
import javax.swing.border.Border;
import javax.swing.event.*;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JMenuBar;
import javax.swing.KeyStroke;
import java.awt.*;
import java.awt.Container.*;
import java.awt.event.*;
import java.util.Date.*;
import java.net.*;
import java.text.MessageFormat;



// Application class
public class PropoGUI extends JFrame
{
    // UI objects
    public DrawArea coordinateArea; 
    public ScribbleArea coordinateArea2; 
    public Toprow toprow;

    // pointer to main program
    Propo controller;

    // get mnemonic from resource bundle
    // e.g getMnemonic("help_sc");
    private Integer getMnemonic( String rbString )
    {
	// see twizz.rug.ac.be/student/prog2.pdf   (1/2003)
	int r =  controller.rb.getString( rbString ).charAt(0) - 'A' + KeyEvent.VK_A ;
        return( new Integer( r ) );
    }

    public void showHighScores( int hiscore )
    {
	if (controller.propoAI.HighScores.size() == 0)
	    {
		// hsEmpty = "The highscore list is empty!"
		JOptionPane.showMessageDialog(null, 
					      controller.rb.getString( "hsEmpty" ), 
					      controller.rb.getString( "hsEmptyTitle" ), 
					      JOptionPane.ERROR_MESSAGE); 
	    }
	else
	    {
		// hsTitle = Highscores
		new HighScores (controller.propoGUI, controller.rb.getString( "hsTitle" ), hiscore, controller );
	    }
    }


    // new game
    public void newGame() 
    {

	controller.propoAI.CurrentGame = new GameStatistics();
	controller.propoAI.NewExercise();
	
	toprow.Reset(); // show "New game"
	update();
    }

    // add game to highscore list if it was a highscore.
    //
    // returns     0 = no hs,
    //             1 = was hs; OK button clicked, hs added,
    //             2 = was hs; cancel clicked, hs not added.
    public int addHighScore()
    {
	GameStatistics record = controller.propoAI.CurrentGame.Clone();
	
	int ranking  = controller.propoAI.HighScores.ranking( record );

	if (ranking != -1)
	    {
		// new highscore:
		// bravo = You have a highscore. \nPlease enter your name. 
		String name = JOptionPane.showInputDialog(controller.rb.getString("bravo"), 
							  controller.playerName);

		if (name != null) // cancel button yields null
		    {
			name.trim();  // leading and trailing whitespaces

			if (name.length() == 0)
			    {
				// save name for next highscore entry
				// anonymousPlayer = "Anonymous"
				controller.playerName = controller.rb.getString("anonymousPlayer");        
			    }
			else
			    {
				controller.playerName = name;
			    }

			record.SetName( controller.playerName );
			controller.propoAI.HighScores.add( record );
			controller.propoAI.HighScores.Save();
       			showHighScores( ranking );
			return ( 1 );
		    }
		return ( 2 );

	    }
	else
	    {
		return ( 0 );  // no hs.
	    }

    }

    // Action: for resetting statistics
    private class ActionReset extends AbstractAction 
    {

	public ActionReset() 
	{
	    // reset = Reset
	    super( controller.rb.getString("reset") );
	    putValue(MNEMONIC_KEY, getMnemonic("reset_sc"));
	}

	public void actionPerformed(ActionEvent e) 
	{
	    newGame();
	}
    }

    // Action: View Highscores
    private class ActionViewHS extends AbstractAction 
    {
	public ActionViewHS() 
	{
	    // viewHighScores = View Highscores
	    super( controller.rb.getString("viewHighScores") );
	    putValue(MNEMONIC_KEY, getMnemonic("viewHighScores_sc"));
	}

	public void actionPerformed(ActionEvent e) 
	{
	    // -1 = do not hilight any entry 
	    showHighScores ( -1 );
	}
    }

    // Action: Quit Propo
    private class ActionQuit extends AbstractAction 
    {
	public ActionQuit() 
	{
	    // quit = Quit
	    super( controller.rb.getString("quit") );
	    putValue(MNEMONIC_KEY, getMnemonic("quit_sc"));
	}

	public void actionPerformed(ActionEvent e) 
	{
	    System.exit(0);
	}
    }

    // Action: Instructions
    private class ActionInstructions extends AbstractAction 
    {
	public ActionInstructions() 
	{
	    super( controller.rb.getString("instructions") );
	    putValue(MNEMONIC_KEY, getMnemonic("instructions_sc"));
	}

	public void actionPerformed(ActionEvent e) 
	{
	    // picture of lightbulb (Renaud Blanchard)
	    //ImageIcon res = new ImageIcon( "ampoule.jpg" );  
	    Object[] args = { Integer.toString( Consts.PLAYTO )  };

	    String msg = MessageFormat.format(controller.rb.getString("instructionMessage"), args);

	    JOptionPane.showMessageDialog(null, 
					  msg, 
					  controller.rb.getString("instructionTitle"), 
					  JOptionPane.INFORMATION_MESSAGE);
					  //res); 

	}
    }

    // Action: About
    private class ActionAbout extends AbstractAction 
    {
	public ActionAbout() 
	{
	    super( controller.rb.getString("about") );
	    putValue(MNEMONIC_KEY, getMnemonic("about_sc"));
	}

	public void actionPerformed(ActionEvent e) 
	{
	    Object[] args = { new String( Consts.VERSION ) };

	    ImageIcon res;
	    URL url = this.getClass().getResource( "bitmaps/icon_64x64.png" ); 
	    res = new ImageIcon( url );         

	    String msg = MessageFormat.format(controller.rb.getString("aboutMessage"), args);
	    JOptionPane.showMessageDialog(null, 
					  msg, 
					  controller.rb.getString("aboutTitle"), 
					  JOptionPane.INFORMATION_MESSAGE,
					  res); 
	}
    }

    // constructor
    public PropoGUI(Propo caller)
    {
	super ( "Propo " + Consts.VERSION );  // give name to window

	controller = caller;
	
	// check for exit button 
        addWindowListener(new WindowAdapter() 
	    {
            public void windowClosing(WindowEvent e) 
		{
		    System.exit(0);
		}
	    });

        // Create menu bar.
        JMenuBar MenuBar = new JMenuBar();
        setJMenuBar(MenuBar);

	Action actionReset = new ActionReset();
	Action actionViewHS = new ActionViewHS();
	Action actionQuit = new ActionQuit();
	Action actionInstructions = new ActionInstructions();
	Action actionAbout = new ActionAbout();

	//actionReset.setMnemonic( KeyEvent.VK_R);

        //Build the first menu.
	// game = Game
        JMenu menuGame = new JMenu( controller.rb.getString("game") );

        menuGame.setMnemonic( getMnemonic("game_sc").intValue() ); 
	JMenuItem itReset = new JMenuItem( actionReset );
	
	menuGame.add ( actionReset );
	menuGame.add ( actionViewHS );
	menuGame.add ( actionQuit );
        MenuBar.add(menuGame);

        JMenu menuHelp = new JMenu( controller.rb.getString("help") );
        menuHelp.setMnemonic(  getMnemonic("help_sc").intValue() ); 
	menuHelp.add ( actionInstructions );
	menuHelp.add ( actionAbout );


        MenuBar.add(menuHelp);





    }

    public void buildUI(Container container) 
    {
        container.setLayout(new BoxLayout( container, BoxLayout.Y_AXIS ));

	// top row for statistics
	toprow = new Toprow( controller );
	container.add( toprow );

	// drawing area
        coordinateArea = new DrawArea( controller );
        container.add( coordinateArea );

	// scribble area
	coordinateArea2 = new ScribbleArea( controller );
        container.add( coordinateArea2 );

	// do the layout. 
	container.doLayout();


    }

    public void update()
    {
	coordinateArea.repaint(); 
	coordinateArea2.repaint(); 
	toprow.repaint();
	
    }


 
}
