
/*
 * WorldAI.java (part of Propo.java)
 * 
 * Copyright (c) 2002, Matias Dahl 
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

import java.awt.*;
import java.awt.geom.*;
import java.util.*;
import java.util.Date.*;
import javax.swing.JOptionPane;



public class WorldAI 
{
    // user statistics
    public GameStatistics CurrentGame;
    public HighScoreData HighScores; 

    // information about last exercise
    boolean LastOK;              // is last exercise OK?
    boolean BullsEye;            // was last point entered a "bull's eye" ?

    // World data
    public World DrawWorld;
    public World ScribbleWorld;
    float currentScaleFactor = (float) 1.0;

    // parameters
    int para_numPoints = 5;                   // number of points
    int para_minDist = 40;                    // minimum distance between any two
                                              // two points
    float para_border = (float) 0.05;         // border for normal points
    float para_sborder = (float) 0.3;         // border for special point
    float para_pRad = (float) 4;              // radius for points

    // reference to caller
    Propo controller;

    // exercise timing
    Date startdate;

    WorldAI(Propo caller)
    {
	// High score data
	HighScores = new HighScoreData(10);
	//HighScores.Load();

	reset();

	BullsEye = false;
	LastOK = true;

	// init world data
	DrawWorld = new World();
	ScribbleWorld = new World();
	
	controller = caller;
    }

    // Update statistics based on a point entered
    public void JudgePoint( PPoint EnteredPoint )
    {
	if (ScribbleWorld.DistanceFromSpecialPoint( EnteredPoint ) <= (2.0f*para_pRad*currentScaleFactor+1) )
	    {
		CurrentGame.ExerciseOK();
		LastOK = true;

		if (ScribbleWorld.DistanceFromSpecialPoint( EnteredPoint ) <= para_pRad*currentScaleFactor)
		    {
			BullsEye = true;
		    }
		else
		    {
			BullsEye = false;
		    }
	    }
	else
	    {
		CurrentGame.ExerciseNotOK();
		LastOK = false;
	    }

    }


    // reset user statistics
    public void reset()
    {
	CurrentGame = new GameStatistics();
    }
    
    // Make a new exercise
    public void NewExercise()
    {
	Dimension wdim = controller.propoGUI.coordinateArea.getSize();

	DrawWorld.Empty ();
	ScribbleWorld.Empty ();

	int enr=0;
	do
	    {
		enr = (int) Math.ceil( Math.random() * 4.0d);
	    }
	while (enr == 0);


	//enr = 3;
	switch (enr)
	    {
	    case 1:
		RandomPoints( wdim );
		break;
	    case 2:
		ProportionalLine( wdim );
		break;
	    case 3:
		PointsOnCircle( wdim );
		break;
	    case 4:
		PointInRectangle( wdim );
		break;

	    default:
		// trouble !
		ProportionalLine( wdim );  
		break;
	    }
		
	rescale( wdim );

    }

    private float ranborder( float p )
    {
	return ( (1-2*p) * (float) Math.random() + p );
    }

    // return a random number in [a,b)
    private float random2( float a, float b )
    {
	float x;
	x = (float) Math.random();
	float f = (b*x + (1-x)*a);
	return ( f );
    }


    // Scale data in ScribbleWorld to suitable size
    private void rescale( Dimension wdim )
    {
	// minimum scale is obtained from minDistance.
	float minscale = (float) para_minDist / ScribbleWorld.minDistance();
	minscale = Math.max(0.5f, minscale);

	// shift points to an origin-centered box
	PPoint re = new PPoint(ScribbleWorld.ULCorner(), 0f );
	re.Scale((float) -1.0);
	ScribbleWorld.Shift( re );

	// .. get coordinates for LR corner of this box
	re = new PPoint( ScribbleWorld.LRCorner(), 0f );
	
	float maxscale =  Math.min((wdim.width-10)/ re.x(), (wdim.height-10)/re.y() );
	maxscale *= (1.0-2.0*para_border);

	if (minscale < maxscale)
	    {
		currentScaleFactor = random2(minscale, maxscale);
		ScribbleWorld.Scale( currentScaleFactor );
	    }
	else
	    {
		currentScaleFactor = 1.0f;
	    }
		
	// Center data in ScribbleWorld 
	re = new PPoint ( ScribbleWorld.LRCorner(), 0f );

	PPoint shift_pnt = new PPoint (0, 0, 0f);

	// strategy 1
	//float x = 3f*para_pRad*currentScaleFactor;
	//shift_pnt.SetX( random2( x, wdim.width-re.x()- x ));
	//shift_pnt.SetY( random2( x, wdim.height-re.y()- x ));

	// strategy 2
	// this command would center the exercise:
	//PPoint shift_pnt = new PPoint((wdim.width-re.x() )/2, (wdim.height-re.y() )/2, 0f);

	// strategy 3
	shift_pnt.SetX( random2( wdim.width * 0.1f,  0.9f * wdim.width-re.x()));
	shift_pnt.SetY( random2( wdim.height * 0.1f , 0.9f * wdim.height-re.y()));

	ScribbleWorld.Shift( shift_pnt );	
    }
    
    private void PointInRectangle ( Dimension wdim )
    {

	int x1 = (int) ( wdim.width * random2 ( 0.1f, 0.5f ) );
	int y1 = (int) ( wdim.height * random2 ( 0.1f, 0.5f ) );
	int x2 = (int) random2 ((float) x1+2f*para_minDist, (float) wdim.width * 0.9f );
	int y2 = (int) random2 ((float) y1+2f*para_minDist, (float) wdim.height * 0.9f );

	DrawWorld.AddPoint (  new PPoint ( x1, y1, para_pRad ));
	DrawWorld.AddPoint (  new PPoint ( x1, y2, para_pRad ));
	DrawWorld.AddPoint (  new PPoint ( x2, y1, para_pRad ));
	DrawWorld.AddPoint (  new PPoint ( x2, y2, para_pRad ));
	ScribbleWorld.AddPoint ( new PPoint ( x1, y1, para_pRad ) );	
	ScribbleWorld.AddPoint ( new PPoint ( x1, y2, para_pRad ) );	
	ScribbleWorld.AddPoint ( new PPoint ( x2, y1, para_pRad ) );	
	ScribbleWorld.AddPoint ( new PPoint ( x2, y2, para_pRad ) );	

	DrawWorld.Add ( new Line2D.Float (x1, y1, x1, y2));
	ScribbleWorld.Add ( new Line2D.Float (x1, y1, x1, y2));
	DrawWorld.Add ( new Line2D.Float (x1, y1, x2, y1));
	ScribbleWorld.Add ( new Line2D.Float (x1, y1, x2, y1));
	DrawWorld.Add ( new Line2D.Float (x2, y1, x2, y2));
	ScribbleWorld.Add ( new Line2D.Float (x2, y1, x2, y2));
	DrawWorld.Add ( new Line2D.Float (x1, y2, x2, y2));
	ScribbleWorld.Add ( new Line2D.Float (x1, y2, x2, y2));

	PPoint sp = new PPoint ( random2(x1+para_minDist,x2-para_minDist), 
				 random2(y1+para_minDist,y2-para_minDist), 
				 para_pRad );
	
	DrawWorld.SetSpecialPoint (  sp );
	ScribbleWorld.SetSpecialPoint ( sp.Clone() );	



    }


    //  ----  random points
    public void RandomPoints( Dimension wdim )
    {
	int x,y;

	// first choose a point the user should point at.
	// this is chosen so that it is more or less in the middle
	int x1 = (int) ((double) wdim.width * ranborder(para_sborder));
	int y1 = (int) ((double) wdim.height * ranborder(para_sborder));

	PPoint sp = new PPoint ( x1, y1, para_pRad );
	DrawWorld.SetSpecialPoint ( sp );
	sp = new PPoint ( x1, y1, para_pRad );
	ScribbleWorld.SetSpecialPoint ( sp );

	// add random dots to world data
	int addedpoints = 0;
	while (addedpoints < para_numPoints)
	    {
		x1 = (int) ((double) wdim.width * ranborder(para_border));
		y1 = (int) ((double) wdim.height * ranborder(para_border));
		PPoint pnt = new PPoint( x1, y1, para_pRad );

		if (DrawWorld.minDistance( pnt ) > para_minDist )
		    {
			addedpoints++;
			DrawWorld.AddPoint ( pnt );
			pnt = new PPoint( x1, y1, para_pRad );
			ScribbleWorld.AddPoint ( pnt );
		    }
	    }

    }

    public void ProportionalLine (Dimension wdim)
    {
	int x1 = (int) ((double) wdim.width * ranborder(para_border));
	int y1 = (int) ((double) wdim.height * ranborder(para_border));
	PPoint A_point = new PPoint(x1, y1, para_pRad);

	int x2, y2;
	PPoint B_point;
	do {
	    x2 = (int) ((double) wdim.width * ranborder(para_border));
	    y2 = (int) ((double) wdim.height * ranborder(para_border));
	    B_point = new PPoint(x2, y2, para_pRad);
	} while (A_point.Distance(B_point) < (para_minDist*3));

	DrawWorld.AddPoint (new PPoint(x1, y1, para_pRad) );
	DrawWorld.AddPoint (new PPoint(x2, y2, para_pRad) );
	ScribbleWorld.AddPoint (new PPoint(x1, y1, para_pRad) );
	ScribbleWorld.AddPoint (new PPoint(x2, y2, para_pRad) );
	DrawWorld.Add ( new Line2D.Float (x1, y1, x2, y2));
	ScribbleWorld.Add ( new Line2D.Float (x1, y1, x2, y2));
		
	// Choose the special point
	int sx, sy;
	if (Math.random() > 0.5)
	    {
		// Set special point anywhere on screen
		sx = (int) ((double) wdim.width * ranborder(para_sborder));
		sy = (int) ((double) wdim.height * ranborder(para_sborder));

		DrawWorld.SetSpecialPoint ( new PPoint (sx, sy, para_pRad) );
		ScribbleWorld.SetSpecialPoint ( new PPoint (sx, sy, para_pRad) );
	    }
	else
	    {
		// Set special point on line
		PPoint dir = new PPoint(x1-x2, y1-y2, 0f);
		float dlen = ranborder(0.2f);
		dir.Scale ( dlen ); 
		
		PPoint A0 = new PPoint (x2, y2, para_pRad);
		A0.Shift( dir );

		DrawWorld.SetSpecialPoint ( A0 );
		ScribbleWorld.SetSpecialPoint ( A0.Clone() );
	    }
    }

    public void PointsOnCircle (Dimension wdim)
    {
	// fix center of circle
	int x1 = (int) ((double) wdim.width * ranborder(para_sborder));
	int y1 = (int) ((double) wdim.height * ranborder(para_sborder));
	PPoint A_point = new PPoint(x1, y1, para_pRad);

	int maxradius = Math.min ( x1, wdim.width - x1 );
	maxradius = Math.min ( maxradius, y1 );
	maxradius = Math.min ( maxradius, wdim.height- y1 );
	
	maxradius *= 0.85f;
	DrawWorld.AddCircle ( x1, y1, maxradius, para_pRad );
	ScribbleWorld.AddCircle ( x1, y1, maxradius, para_pRad );

	if (Math.random() < 0.5)
	    {
		PPoint p1, p2;

		// add center of worlds
		DrawWorld.AddPoint ( A_point );	
		ScribbleWorld.AddPoint ( A_point.Clone() );	

		// fix and draw one point on the perimeter of the circle
		float sangle = random2( 0f, (float) (2* Math.PI));
	
		p1 = new PPoint((float) (x1 + maxradius * Math.cos((double) sangle)),
				(float) (y1 + maxradius * Math.sin((double) sangle)), 
				para_pRad);

		DrawWorld.AddPoint ( p1 );
		ScribbleWorld.AddPoint ( p1.Clone() );

		// .. and another point
		sangle += random2( (float) (10f/180f * Math.PI), 
				   (float) (2*Math.PI-10f/180f*Math.PI));
	
		p2 = new PPoint((float) (x1 + maxradius * Math.cos((double) sangle)),
				(float) (y1 + maxradius * Math.sin((double) sangle)), 
				para_pRad);
		DrawWorld.SetSpecialPoint ( p2 );
		ScribbleWorld.SetSpecialPoint ( p2.Clone() );

		// draw lines from center of the two points
		DrawWorld.Add ( new Line2D.Float (x1, y1, p1.x(), p1.y()));
		DrawWorld.Add ( new Line2D.Float (x1, y1, p2.x(), p2.y()));
		ScribbleWorld.Add ( new Line2D.Float (x1, y1, p1.x(), p1.y()));		

	    }
	else
	    {
		PPoint p1;

		float sangle = random2( 0f, (float) (2* Math.PI));
		float srad = random2( 0f, (float) maxradius);
	
		p1 = new PPoint((float) (x1 + srad * Math.cos((double) sangle)),
				(float) (y1 + srad * Math.sin((double) sangle)), 
				para_pRad);
		DrawWorld.SetSpecialPoint ( p1 );
		ScribbleWorld.SetSpecialPoint ( p1.Clone() );

	    }

    }

}

