
/*
 * World.java (part of Propo.java)
 * 
 * Copyright (c) 2002, Matias Dahl 
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

import java.awt.*;
import java.awt.geom.*;
import java.awt.BasicStroke.*;
import java.util.*;

// Application class

//
// important:
//   - always add the special point first to a world.
//
public class World
{
    float F_MIN = (float) java.lang.Float.MIN_VALUE;
    float F_MAX = (float) java.lang.Float.MAX_VALUE;

    // world data
    PPoint SpecialPoint;  // special point the user should point at
    java.util.LinkedList pdata;  // point world
    java.util.LinkedList ldata;  // line world
    java.util.LinkedList cdata;  // circle world


    // -- General routines

    World() 
    {
	pdata = new java.util.LinkedList();
	ldata = new java.util.LinkedList();
	cdata = new java.util.LinkedList();
	SpecialPoint = new PPoint();
    }

    // Add a point to the world
    public void AddPoint(PPoint q)
    {
	pdata.add(q);
    }

    // Add a line to the world
    public void Add(Line2D.Float q)
    {
	ldata.add(q);
    }

    // Add a line to the world
    public void Add(Ellipse2D.Float q)
    {
	cdata.add(q);
    }

    // Add circle to the world
    public void AddCircle(float x, float y, float r, float linewidth)
    {
	cdata.add(new Ellipse2D.Float(x-r,y-r,2*r, 2*r));
    }


    // Remove all objects from world
    public void Empty ()
    {
	pdata.clear();
	ldata.clear();
	cdata.clear();
    }


    // ** Routines for special point

    // set special point to q
    public void SetSpecialPoint(PPoint q)
    {
	SpecialPoint = q;
    }


    // ** Distance properties of World

    // return length from a point to the special point
    public float DistanceFromSpecialPoint( PPoint pp)
    {
	return (pp.Distance( SpecialPoint ));
    }

    // Measure the shortest distance from pp to any point in world
    // including the special point. 
    public float minDistance( PPoint pp )
    {
	float md, tt;

	md = this.DistanceFromSpecialPoint( pp );

	for (Iterator i = this.pdata.iterator(); i.hasNext();) 
	    {
		md = Math.min (pp.Distance( (PPoint) i.next()), md);
	    }

	return ( md );
    }

    // Measure the shortest distance between any two points in the world
    public float minDistance( )
    {
	float ans = (float) F_MAX;
	float tt;

	for (ListIterator i = this.pdata.listIterator(); i.hasNext();) 
	    {
		PPoint basePoint = new PPoint( (PPoint) i.next(), 0f );
		
		ans = Math.min( ans, this.DistanceFromSpecialPoint( basePoint ) );

		int backCounter=0;

		while (i.hasNext())
		    {
			PPoint runPoint = new PPoint( (PPoint) i.next(), 0f );
			backCounter++;

			ans = Math.min( ans, basePoint.Distance( runPoint ) );

		    }

		// restore iterator 
		while (backCounter!=0)
		    {
			i.previous();
			backCounter--;
		    }
		
	    }
	return ( ans );
    }


    // Return coordinates to lower right corner as PPoint
    public PPoint LRCorner()
    {
	PPoint ans = new PPoint( SpecialPoint, 0f ); 
	PPoint temp;

	for (Iterator i = pdata.iterator(); i.hasNext();) 
	    {
		temp = (PPoint) i.next();
		ans.SetX ( Math.max(temp.x(), ans.x() ) );
		ans.SetY ( Math.max(temp.y(), ans.y() ) );
	    }


	Ellipse2D.Float tempc;
	Rectangle tbound;

	for (Iterator j = cdata.iterator(); j.hasNext();) 
	    {
		tempc = (Ellipse2D.Float) j.next();
		tbound = tempc.getBounds();
		
		ans.SetX ( Math.max (tbound.x + tbound.width, ans.x() ));
		ans.SetY ( Math.max (tbound.y + tbound.height, ans.y() ));
	    }

	// Since lines start and end with a PPoint, they do not need to be 
	// checked.

	return (ans);
    }


    // Return coordinates to upper left corner as PPoint
    public PPoint ULCorner()
    {
	PPoint ans = new PPoint( SpecialPoint, 0f );
	PPoint temp;

	for (Iterator i = pdata.iterator(); i.hasNext();) 
	    {
		temp = (PPoint) i.next();
		ans.SetX ( Math.min(temp.x(), ans.x()) );
		ans.SetY ( Math.min(temp.y(), ans.y()) );
	    }


	Ellipse2D.Float tempc;
	Rectangle tbound;

	for (Iterator j = cdata.iterator(); j.hasNext();) 
	    {
		tempc = (Ellipse2D.Float) j.next();
		tbound = tempc.getBounds();

		ans.SetX ( Math.min (tbound.x, ans.x() ));
		ans.SetY ( Math.min (tbound.y, ans.y() ));
	    }

	return (ans);
    }


    // -- Operations
    
    // Scale all objects in World by factor
    public void Scale (float factor)
    {
	for (Iterator i = pdata.iterator(); i.hasNext();) 
	    {
		PPoint q = (PPoint) i.next();
		q.Scale( factor );
	    }

	for (Iterator j = ldata.iterator(); j.hasNext();) 
	    {
		Line2D.Float q = (Line2D.Float) j.next();

		double  a = q.getX1();
		double  b = q.getY1();
		double  c = q.getX2();
		double d = q.getY2();
		q.setLine (a * factor, b * factor, c*factor, d*factor);

	    }

	for (Iterator k = cdata.iterator(); k.hasNext();) 
	    {
		Ellipse2D.Float q = (Ellipse2D.Float) k.next();

		double  a = q.height;
		double  b = q.width;
		double  c = q.x;
		double d = q.y;
		q.setFrame (c * factor, d * factor, b*factor, a*factor);

	    }

	SpecialPoint.Scale( factor );
    }


    // Shift all objects in World by spoint
    public void Shift (PPoint spoint)
    {
	for (Iterator i = pdata.iterator(); i.hasNext();) 
	    {
		PPoint q = (PPoint) i.next();
		q.Shift( spoint );
	    }

	for (Iterator j = ldata.iterator(); j.hasNext();) 
	    {
		Line2D.Float q = (Line2D.Float) j.next();

		double  a = q.getX1();
		double  b = q.getY1();
		double  c = q.getX2();
		double d = q.getY2();
		q.setLine (a + spoint.x(), b + spoint.y(), c + spoint.x(), d + spoint.y());

	    }


	for (Iterator k = cdata.iterator(); k.hasNext();) 
	    {
		Ellipse2D.Float q = (Ellipse2D.Float) k.next();

		double  a = q.height;
		double  b = q.width;
		double  c = q.x;
		double d = q.y;
		q.setFrame (c + spoint.x(), d + spoint.y(), b, a);

	    }


	SpecialPoint.Shift( spoint );
    }


    // -- Drawing routines

    // Draw whole world
    public void DrawAll (Graphics g)
    {
	this.DrawObjects ( g );
	this.DrawSpecialPoint( g );

    }

    // Draw the special point
    public void DrawSpecialPoint (Graphics g)
    {
	SpecialPoint.Draw ((Graphics2D) g, Color.red );
    }

    // Draw all objects
    public void DrawObjects (Graphics g)
    {
	Graphics2D g2 = (Graphics2D) g;

	g2.setStroke( new BasicStroke (2.0f) );
	g2.setColor(Color.black);

	for (Iterator i = pdata.iterator(); i.hasNext();) 
	    {
		PPoint cpoint = (PPoint) i.next();
		cpoint.Draw( g2, Color.black  );
	    }

	for (Iterator j = ldata.iterator(); j.hasNext();) 
	    {
		Line2D.Float cline = (Line2D.Float) j.next();
		g2.draw( cline );
	    }

	for (Iterator k = cdata.iterator(); k.hasNext();) 
	    {
		Ellipse2D.Float ccircle = (Ellipse2D.Float) k.next();
		g2.draw( ccircle );
	    }
    }

}

