
function RNew = subrelationCT(Q1,Q2,R);
% 
% function RNew = subrelationCT(Q1,Q2,R);
% Lasse Leskela 2008-10-01
%
% Compute one step of the subrelation algorithm for
% rate kernels of continuous-time finite-state Markov
% processes [1, Theorems 4.11 and 4.14].
%
% Input:
% - Q1: S1xS1 rate matrix (diagonal values are ignored)
% - Q2: S2xS2 rate matrix (diagonal values are ignored)
% - R:  S1xS2 array of logicals, R(x,y) = true if x and y are related
%
% Output:
% - RNew: S1xS2 array of logicals
%
% References:
% [1] L. Leskela. Stochastic relations of random variables and processes.
% http://arxiv.org/abs/0806.3562
%

RNew = R;
S1 = size(R,1);
S2 = size(R,2);

% Replace the diagonal values of Q1 and Q2 by zero.
Q1 = Q1.*(1-eye(size(Q1)));
Q2 = Q2.*(1-eye(size(Q2)));

for x = 1:S1,
  xRight = rightConjugate(x,R);

  % i indexes the elements y in S2 that are related to x
  for i = 1:length(xRight),
    y = xRight(i);
    % The states reachable from x by Q1, which are not related to y.
    K1 = setdiff(find(Q1(x,:)>0),leftConjugate(y,R));

    % j indexes the nonempty subsets of K1
    for j = 1:(power(2,length(K1))-1),
      U = getSubset(K1,j);
      URight = rightConjugate(U,R);
      if sum(Q1(x,U)) > sum(Q2(y,URight)) + eps,
        RNew(x,y) = false;
        break;
      end;
    end;
    if RNew(x,y),
      % The states reachable from y by Q2, which are not related to x.
      K2 = setdiff(find(Q2(y,:)>0),rightConjugate(x,R));

      % j indexes the nonempty subsets of K2
      for j = 1:(power(2,length(K2))-1),
        V = getSubset(K2,j);
        VLeft = leftConjugate(V,R);
        if sum(Q1(x,VLeft)) < sum(Q2(y,V)) - eps,
          RNew(x,y) = false;
          break;
        end;
      end;
    end;
  end;
end;
