%
% example.m
% Lasse Leskela 2008-10-01
%
% This example illustrates the subrelation algorithm in [1, Theorem 4.14] 
% applied to a simple queueing system.
%
% Consider a parallel system of two queues, where the interarrival and service
% times for both queues are independent and exponentially distributed.
% The buffers of both queues are finite, and customers finding a full buffer
% leave the system immediately. To study the time-dependent behavior of the
% system, we shall compare two models with different parameters indexed by
% m = 1,2 so that in model m:
% - la{m}(i) is the arrival rate to queue i
% - mu{m}(i) is the service rate of queue i
% - K{m}(i)  is the buffer capacity of queue i
%
% The following program computes the subrelation algorithm for the 
% relations RCoord and RSum defined by
% 
%     RCoord = { (x,y): x(1) <= y(1) and x(2) <= y(2) },
%
%     RSum   = { (x,y): x(1) + x(2) <= y(1) + y(2) }.
%
% References:
% [1] L. Leskela. Stochastic relations of random variables and processes.
% http://arxiv.org/abs/0806.3562
%

% Parameters for model 1.
la{1} = [1 1];
mu{1} = [1 1];
K{1}  = [5 3];

% Parameters for model 2.
la{2} = [2 1];
mu{2} = [1 1];
K{2}  = [5 4];

% State enumeration.
for m=1:2,
  s = 0;
    for x1=0:K{m}(1),
      for x2=0:K{m}(2),
        s = s + 1;
        state{m}{s} = [x1 x2];
      end;
    end;
  S{m} = s;
end;

% Construction of the rate matrices.
e{1} = [1 0];
e{2} = [0 1];
for m=1:2,
  Q{m} = zeros(S{m},S{m});
    for sx = 1:S{m}
      x = state{m}{sx};
      for sy = 1:S{m}
        y = state{m}{sy};
        for i=1:2
          if all(y == x + e{i})
            Q{m}(sx,sy) = la{m}(i);
          elseif all(y == x - e{i})
            Q{m}(sx,sy) = mu{m}(i);
          end;
      end;
    end;
  end;
  Q{m} = Q{m} - diag(sum(Q{m},2));
end;

% Construction of the relations.
for sx=1:S{1},
  x=state{1}{sx};
  for sy=1:S{2},
    y=state{2}{sy};
    RCoord(sx,sy) = (all(x <= y));
    RSum(sx,sy) = (sum(x) <= sum(y));
  end;
end;

disp(' ');
disp('1. Subrelation algorithm for the relation RCoord:');
disp(' ');
disp('Information about RCoord:');
relationInfo(RCoord);
disp(' ');
disp(['Running algorithm...']);
[RCoordStar iter] = maximalSubrelationCT(Q{1},Q{2},RCoord);
disp(['Algorithm finished after ' num2str(iter) ' iterations.']);
disp(' ');
disp('Information about RCoordStar:');
relationInfo(RCoordStar);
disp(' ');

disp('2. Subrelation algorithm for the relation RSum:');
disp(' ');
disp('Information about RSum:');
relationInfo(RSum);
disp(' ');
disp(['Running algorithm...']);
[RSumStar iter] = maximalSubrelationCT(Q{1},Q{2},RSum);
disp(['Algorithm finished after ' num2str(iter) ' iterations.']);
disp(' ');
disp('Information about RSumStar:');
relationInfo(RSumStar);
disp(' ');

figure;
subplot(2,2,1);
plotRelation(RCoord);
title('R_{coord}');
subplot(2,2,2);
plotRelation(RCoordStar);
title('R_{coord}^*');
subplot(2,2,3);
plotRelation(RSum);
title('R_{sum}');
subplot(2,2,4);
plotRelation(RSumStar);
title('R_{sum}^*');
