from itertools import groupby
import math
from sage.combinat import q_analogues
from sage.combinat.q_analogues import q_int
from sage.combinat.q_analogues import q_factorial

# Symmetric functions over Q(q,t)

Symqt = SymmetricFunctions(QQ['q','t'].fraction_field())
(q,t) = Symqt.base_ring().gens()
e = Symqt.e()
s = Symqt.s()
h = Symqt.h()
p = Symqt.p()
m = Symqt.m()
Ht=Symqt.macdonald(t=1).Ht()

# Various q-numbers
def flagsize(part):
    return prod(sum([q^i for i in range(pa)]) for pa in part)
def psp(part):
    return prod([1-q^(pa) for pa in part])
def qfact(part):
    return prod(q_factorial(pa) for pa in part)
def ccoeff(part):
    return prod(prod([q^j-1 for j in range(1,pa+1)]) for pa in part)

# The source for class RPF is Greg Warrington's rational parking function code in https://gswarrin.w3.uvm.edu/research/RPF.sws
# Rational Parking Function class
# 
# Basic object is a NE-lattice path
# - paths run from (0,0) to (b,a)
# - if Dyck, then stay above sy=rx
# - labels are not actually included as part of the class, 
#   passed in when needed

class RPF(tuple):
    """ lattice paths in an a (height) by b (width) rectangle
    """
    def __new__(cls,ne=(),a=1,b=1,r=1,s=-1):
        return tuple.__new__(cls,ne)

    def __init__(self,ne=[],a=1,b=1,r=1,s=-1):
        self.ne = tuple(ne)
        self.a = a # Number of North steps
        self.b = b # Number of East steps
        self.r = r # Change in level for a North step
        self.s = s # change in level for an East step

    ##########################################################
    # generating paths of particular types
    ##########################################################
    @classmethod
    def pats(cls,dyckonly,a=1,b=1,r=1,s=-1):
        """ Generate rectangular paths of size (a,b)
            If dyckonly = True, then stay above sy=rx.
        """
        if a == 0 or b == 0:
            return [RPF((),a,b,r,s)]

        cpat = []  # current path we're creating
        ta = a     # number of Ns we have left to use
        tb = b     # number of Es we have left to use
        clvl = 0   # current level of end of path
        clist = [] # list of patterns we've found

        # keeps track of how much g-vector can decrease 
        cls._recpats(a,b,r,s,dyckonly,cpat,ta,tb,clvl,clist)
        return clist
    @classmethod
    def dyckpaths(cls,m,n): return cls.pats(True, m, n, n, -m)
    @classmethod
    def _recpats(cls,a,b,r,s,dyckonly,cpat,ta,tb,clvl,clist):
        """ Generate rectangular paths of size (a,b)
            If dyckonly = True, then stay above ay=bx.
            Called by pats
        """
        # if only want Dyck paths, need to check that level >= 0
        if dyckonly and clvl < 0:
            return

        # we've reached (a,b)
        if(ta == 0 and tb == 0):
            clist.append(RPF(cpat,a,b,r,s))
            return

        # try to add an N step
        if ta > 0:
            cpat.append('N')
            cls._recpats(a,b,r,s,dyckonly,cpat,ta-1,tb,clvl + r,clist)
            cpat.pop()

        # try to add an E step
        if tb > 0:
            cpat.append('E')
            cls._recpats(a,b,r,s,dyckonly,cpat,ta,tb-1,clvl + s,clist)
            cpat.pop()
            
            
    ##########################################################
    # levels and g-vector functions
    ##########################################################
    def path_to_gv(self):
        """ Return g-vector (i.e., area vector) from NE sequence
        """
        r = self.r
        s = self.s

        # make vector of minimum non-negative coords
        minpos = [(r*i)%abs(s) for i in range(self.a)]
        tv = ''.join(self).split('N')
        # get lengths of sequence of Es
        tv = list(map(lambda x: len(x), tv))
        #return(tv)
        # count number of squares in each row between path and diagonal
        return tuple(map(lambda x: (r*x+s*sum(tv[:x+1])-minpos[x])/abs(s),range(len(tv)-1)))

    def isdyck(self):
        """ Return True if an (r,s)-Dyck path
        """
        return self.mindiag() >= 0

    def levels(self):
        """ Return levels of steps in NE path

        Note that the N steps are weighted by self.r and
            the E steps are weighted by self.s.
        Levels are according to South/West endpoint of step
        """
        if len(self) == 0:
            return []
        ans = []
        cur = 0
        for x in self:
            ans.append(cur)
            if x == 'N':
                cur = cur + self.r
            else:
                cur = cur + self.s
        return ans

    def mindiag(self):
        """ Return minimum diagonal hit by the path.
        """
        if len(self) == 0:
            return 0
        else:
            return min(self.levels())

    ##########################################################
    # basic statistics
    ##########################################################
    def area(self):
        """ Compute the area of an r/s-Dyck path
        """
        if not self.isdyck():
            print("Area not programmed for non-r/s-Dyck")
            return -1
        return sum(self.path_to_gv())

    def ptn_area(self):
        """ Compute the area above a path
        """
        ans = 0
        for i in range(self.a + self.b - 1):
            if self[i] == 'E':
                for j in range(i+1,self.a + self.b):
                    if self[j] == 'N':
                        ans = ans + 1
        return ans
    
    def horizontal_steps(self):
        "Compute vector of horizontal steps"
        return [sum(1 for _ in group) for _, group in groupby(self)][1::2]

##########################################################
# Master symmetric functions
##########################################################
def pathswithareas(m,n,k):
    mypats = RPF.dyckpaths(k*m,k*n)
    ans=0
    for p in mypats: 
        f=e[0]
        for step in p.horizontal_steps(): f=f*e[step]
        ans+=q^(p.area())*f
    return(ans)

def pathswithareas_h(m,n,k):
    mypats = RPF.dyckpaths(k*m,k*n)
    ans=0
    for p in mypats: 
        f=h[0]
        for step in p.horizontal_steps(): f=f*h[step]
        ans+=q^(p.area())*f
    return(ans)

def plethysm(f, m, n):
    deg=f.degree()
    ans=0
    coeffs=[e(f).coefficient(part) for part in Partitions(deg)]
    E=[0 for k in range(1,deg+1)]
    for k in range(1,deg+1):
        E[k-1]=pathswithareas(m,n,k)
    for c in range(len(coeffs)): 
        ans+=coeffs[c]*prod([E[p-1] for p in Partitions(deg)[c]])
    return ans
