function [Radius,m,Err] = joraderr(A,dA,Delta,M,p,Err,Display)

% function [Radius,m,Err] = joraderr(A,dA,Delta,M,p,Err,Display)
% This functions estimates the joint spectral radius of the matrices
% contained in A = [A1, A2, A3, ..., AN]
% dA is an estimate of the norm of the error in the matrices in A
% If dA = [dA(1)], the errors are assumed to be dA(1)*norm(Ai,p)
% Delta = [Delta(1),Delta(2)] gives bounds for the length of the 
%    interval in which the joint spectral radius lies; if only 
%    one number is given, Delta(2) = Delta(1), default 1/100
% M is the maximum number of evaluations of norms and eigenvalues
% p gives the norm to be used
% Err = [Err(1), Err(2)] gives the relative errors in evaluating products
%     and norms. Default   [2^(-50)*n,2^(-49)*n] if p==1 or p==inf
%                     and  [2^(-50)*n^2,2^(-49)*n^2] otherwise.
% Display = 1 shows progress in the calculations, default = 0.
% Radius = [lower bound for the JSR, upper bound for the JSR]
% m = [optimal number of factors, largest number of factors,
%         number of evaluations of norms and eigenvalues, maximum number of 
%           matrices considerd at each stage]
%           The program calls lowspera.m
%
% G. Gripenberg 5.8.1996
% Reference: G. Gripenberg, Computing the joint spectral radius, Linear
%      Algebra Appl.  234 (1996) 43--60.

if nargin < 2
  dA = [];
end


if nargin < 3
  Delta = 10^(-2);
end

Delta = Delta(:);
if size(Delta)*[1;0] == 1
   Delta(2) = Delta(1);
end



if nargin <4
  M = 1000;
end

if nargin < 5
  p = 2;
end

if nargin < 6
  Err = [];
end

if nargin < 7
  Display = 0;
end

[ma,na] = size(A);


if (na > ma & rem(na,ma) == 0)
  N = fix(na/ma);
  n = ma;
elseif (ma > na & rem(ma,na) == 0)
  A = A';
  N = fix(ma/na);
  n = na;
else
 disp('The matrix has wrong dimensions');
 return;
end

if  size(Err)*[1;1] < 3
  if p==1 | p==inf
    Err = [2^(-50)*n,2^(-49)*n];
  else
    Err = [2^(-50)*n^2,2^(-49)*n^2];
  end
end


dA = dA(:);
if  size(dA)*[1;0] == n
  proportional = 0;
else 
  proportional = 1;
  if size(dA)*[1;0] == 0
    dA = eps;
  end
end





X = [];
LowRadius = 0;
for i = 1:N
  Y = A(:,(i-1)*n+1:i*n);
  NORM(i) = norm(Y,p)*(1+Err(2));
  UpBound(i) = NORM(i);
  if proportional
    NormErr(i) = dA(1) * NORM(i);
  else
    NormErr(i) = dA(i);
  end
  LowRadius = max(LowRadius,lowspera(Y,NormErr(i),p,Err,LowRadius));
  X = [X;Y];
end  

dA = NormErr;
NormA = NORM;
HighRadius = max(NORM+NormErr);
mm = N; m = 1; JJ = N; MaxJJ = N;
BestPower = 1;

while mm < M & HighRadius > LowRadius + Delta(2)
  m = m + 1;
  OldHighRadius = HighRadius;
  OldLowRadius = LowRadius;
  HighRadius = LowRadius;   
  NewJJ = 0;
  for k = 1:JJ
    for i = 1:N
     NJJN = NewJJ + 1;
     XX = X((k-1)*n+1:k*n,:)*A(:,(i-1)*n+1:i*n);
     NewNormErr(NJJN) = (NORM(k)*(Err(1)*NormA(i)+dA(i)) + ...
         NormErr(k)*(NormA(i)+dA(i)));
     NewNORM(NJJN) = norm(XX,p)*(1+Err(2));
     NewUpBound(NJJN) = min(UpBound(k),(NewNORM(NJJN)+NewNormErr(NJJN))^(1/m));
     LowRadius = max(LowRadius, ...
         lowspera(XX,NewNormErr(NJJN),p,Err,LowRadius^m)^(1/m));
     if NewUpBound(NJJN) > LowRadius + Delta(1)
        NewJJ = NewJJ + 1;
        HighRadius = max(HighRadius,NewUpBound(NJJN));
        NewX((NewJJ-1)*n+1:NewJJ*n,:) = XX;
     end
    end
  end
  X = NewX;
  HighRadius = min(OldHighRadius,max(HighRadius,LowRadius+Delta(1)));
  mm = mm+JJ*N;
  NORM = NewNORM;
  UpBound = NewUpBound;
  NormErr = NewNormErr;
  JJ = NewJJ;
  MaxJJ = max(JJ,MaxJJ);
  if HighRadius-LowRadius < OldHighRadius-OldLowRadius
     BestPower = m;
  end
  if Display
     disp([BestPower,m,mm,JJ]);
     disp([LowRadius,HighRadius]);
  end
end

m = [BestPower,m,mm,MaxJJ];

Radius = [LowRadius,HighRadius];


